/*
 i-net software provides programming examples for illustration only, without warranty
 either expressed or implied, including, but not limited to, the implied warranties
 of merchantability and/or fitness for a particular purpose. This programming example
 assumes that you are familiar with the programming language being demonstrated and
 the tools used to create and debug procedures. i-net software support professionals
 can help explain the functionality of a particular procedure, but they will not modify
 these examples to provide added functionality or construct procedures to meet your
 specific needs.
  
  i-net software 1998-2013

*/
using System;

namespace Inet.Viewer.Data
{
    using Inet.Viewer.Resources;


    /// <summary>
    /// Abstract super class of a prompt value  
    /// </summary>
    [System.SerializableAttribute()]
    public abstract class PromptValue
    {
        /// <summary>
        /// decription for the prompt value </summary>
        private string description = "";

        /// <summary>
        /// value type of this prompt </summary>
        private int type;

        /// <summary>
        /// if true show only the decription for the prompt value </summary>
        private bool onlyDescription;

        /// <summary>
        /// Creates a prompt values of the given type initialized with the default value of its type </summary>
        /// <param name="type"> type of this default value. </param>
        public PromptValue(int type)
        {
            this.type = type;
            InitValue();
        }

        /// <summary>
        /// Creates a prompt values of the given type initialized with the default value of its type </summary>
        /// <param name="type"> type of this default value. </param>
        /// <param name="description"> decription of this promt </param>
        public PromptValue(string description, int type)
        {
            this.description = description;
            this.type = type;
        }

        /// <summary>
        /// Returns the value of this prompt as string. </summary>
        /// <returns> the value of this prompt as string. </returns>
        internal abstract string ValueString { get; }

        /// <summary>
        /// Returns the value of this prompt as string. </summary>
        /// <param name="promptValue"> the given promptValue. </param>
        /// <returns> the value of this prompt as string. </returns>
        public static string GetValueString(PromptValue promptValue)
        {
            if (promptValue == null)
            {
                return strings.Prompt_None;
            }
            else
            {
                return promptValue.ValueString;
            }
        }

        /// <summary>
        /// Returns a string representation of the value of this prompt. </summary>
        /// <returns> a string representation of the value of this prompt. </returns>
        public abstract string StringRepresentation { get; }

        /// <summary>
        /// Returns the value of this prompt - its type will depend on which value type this PromptValue has:
        /// String for STRING<br/>
        /// Double for NUMBER and CURRENCY<br/>
        /// Boolean for BOOLEAN<br/>
        /// Date for DATE and DATETIME<br/>
        /// Time for TIME<br/>
        /// byte[] for BINARY </summary>
        /// <returns> the value of this prompt - its type will depend on which value type this PromptValue has. </returns>
        public abstract object Value { get; }

        /// <summary>
        /// initializes the prompt value.
        /// </summary>
        internal abstract void InitValue();

        /// <summary>
        /// Creates a formated string representation of this prompt value with its decription and value. </summary>
        /// <returns> String representation </returns>
        public override string ToString()
        {
            string value = ValueString;
            string result = "";

            if (this.Description != null && this.Description.Length > 0)
            {
                if (!this.OnlyDescription)
                {
                    result = value + " - ";
                }
                result += this.Description;
            }
            else
            {
                return value;
            }
            return result;
        }

        /// <summary>
        /// Get description of the prompt value </summary>
        /// <returns> decription of the prompt value </returns>
        public virtual string Description
        {
            get
            {
                return description;
            }
            set
            {
                this.description = value;
            }
        }


        /// <summary>
        /// Get value type of the prompt, see the value type constants in <seealso cref="PromptData"/> </summary>
        /// <returns> value type of the prompt, see the value type constants in <seealso cref="PromptData"/> </returns>
        public virtual int Type
        {
            get
            {
                return type;
            }
        }

        /// <summary>
        /// If true only the decription will be displayed in the report dialog. 
        /// (only necessary if this prompt value represents a default value) </summary>
        /// <returns> whether only description is displayed </returns>
        public virtual bool OnlyDescription
        {
            get
            {
                return onlyDescription;
            }
            set
            {
                this.onlyDescription = value;
            }
        }
        /// <summary>
        /// Used for showing the values in combo boxes
        /// </summary>
        public string DisplayString
        {
            get
            {
                return this.ToString();
            }
        }

        /// <inheritDoc/>
        public override bool Equals(object obj)
        {
            return obj is PromptValue && StringRepresentation == ((PromptValue)obj).StringRepresentation;
        }

        /// <inheritDoc/>
        public override int GetHashCode()
        {
            if (StringRepresentation == null)
            {
                return 0;
            }
            return StringRepresentation.GetHashCode();
        }
    }

}